# Copyright 2011-2012 Michael Pechner NE6RD
# Distributed under the terms of the MIT License:
# http://www.opensource.org/licenses/mit-license

import sys
import os
from optparse import OptionParser
import re

"""
Conversion from DMS to Decimal Degree

Given a DMS (Degrees, Minutes, Seconds) coordinate such as W8743'41", it's trivial to convert it to a number of decimal degrees using the following method:
    * Calculate the total number of seconds, 43'41" = (43*60 + 41) = 2621 seconds.
    * The fractional part is total number of seconds divided by 3600. 2621 / 3600 = ~0.728056
    * Add fractional degrees to whole degrees to produce the final result: 87 + 0.728056 = 87.728056
    * Since it is a West longitude coordinate, negate the result.
    * The final result is -87.728056.


Type   Dir.   Sign    Test
  Lat.   N      +       > 0
  Lat.   S      -       < 0
  Long.  E      +       > 0
  Long.  W      -       < 0

Altitude is A Ft / 3.2808FT/M = A in Meters

"""
callsign=''
latRE = re.compile('(\d\d)(\d\d)\.(\d\d)(\w)')  
lonRE = re.compile('(\d{2,3})(\d\d)\.(\d\d)(\w).(\d{3})')
AltRE = re.compile('^(\d+)\D.*')
#plane logo
plane_img = 'http://dl.dropbox.com/u/15037198/plane.png'
# parachute Logo
parachute_img = 'http://farm5.static.flickr.com/4031/4470630165_85e7b8f701_o.png'

min_altitude = 100

fly_alt=200
fly_tilt=90
fly_heading=0
fly_range=1000


dataRE = re.compile('(\w{3,7}\-\d+|\w{3,7}).*:/([\d.]+)h([\d.]+[NS])/([\d.]+[EW].\d{3})/\d+/A=(\d+)(.*)$')
#                     call1            tm2                 lat3         lon4               alt5      misc

lastHour=-1
starttime=None
kmlfh=None
last_alt = 0

def lat(lat):
    global latRE
    mat = latRE.match(lat)
    if mat:
        latF = float(mat.group(1)) + (((int(mat.group(2))*60 ) + int(mat.group(3))) / 3600.0 )    
        if mat.group(4) == 'S':
            return  latF * -1.0
        return latF
    
# parse the Lonitude string conerting from Degrees Minutes Secods to Decimal degree format
def lon(lon):
    global lonRE
    mat = lonRE.match(lon)
    if mat:
        lonF = float(mat.group(1)) + (((int(mat.group(2))*60 ) + int(mat.group(3)) + float('0.'+mat.group(5)))/3600)    
        if mat.group(4) == 'W':
            return  lonF * -1.0
        return lonF

#process a line and print it out
def processline(data):
    global dataRE, callsign, lastHour, starttime, kmlfh, last_alt, AltRE, min_altitude
    match =  dataRE.match(data)
    if not match:
        return 

    #print str(match.groups())+"\n"

    latitude = lat(match.group(3))
    longitude = lon(match.group(4))
    hour = int(match.group(2)[:2])
    minute = int(match.group(2)[2:4])
    second = int(match.group(2)[4:])
    altitudeFt = match.group(5)
    data = match.group(6)
    altitude = int(int(altitudeFt)/3.2808)
   
    if  altitude == None or altitude <= min_altitude  :
        return

    timeStr = '%4s-%2s-%2sT%2d:%2d:%2dZ'%(starttime[4:], starttime[:2], starttime[2:4], hour, minute, second)

    print 'call:'+  callsign + ' time:'+ timeStr + ' lat:'+  str(latitude) + ' lon:' +  str(longitude) + ' alt:' + str(altitude) + ' date:'+ data + "\n" 
    
    kmlfh.write("<Placemark>\n")
    kmlfh.write ('<LookAt> <longitude>%s</longitude> <latitude>%s</latitude>  <altitude>%s</altitude>  <range>300</range>   <tilt>45</tilt> <heading>270</heading> <altitudeMode>absolute</altitudeMode> </LookAt> '%(str(longitude),str(latitude), str(altitude))+"\n")
    kmlfh.write("\t<TimeStamp><when>%s</when></TimeStamp>\n"%timeStr)

    if altitude > last_alt :
        kmlfh.write("\t\t\t<styleUrl>#myPlaneStyles</styleUrl>\n")
    else:
        kmlfh.write("\t\t\t<styleUrl>#myDefaultStyles</styleUrl>\n")

    last_alt = altitude

    
    desc = 'Altitude:'+altitudeFt+'ft'
    xtra_desc = tactical_data(data)
    if len(xtra_desc) != 0:
        desc+="\n"+xtra_desc
    kmlfh.write("<description>%s</description>\n"%(desc))
    kmlfh.write("\t<Point>  <altitudeMode>absolute</altitudeMode><coordinates>%s,%s,%s</coordinates> </Point>\n"%(longitude,latitude,altitude))
    kmlfh.write("</Placemark>\n")
    
#
# return the extra data used in the info balloon
def tactical_data(data):
    if data.startswith('PULSEOX'):
        return ''

    val = data.split()
    #print "Tactical:"+str(val) + "  " + data+"\n"
    try:
        if float(val[1]) < 200 and float(val[3]) < 101:
                return " HR: %s SPO2: %s "%(val[1], val[3])
    except Exception, e:
        print val, e

# The first valid data point it finds is the used to calulate the point of view for the "Fly To"
def processFly(data):
    global dataRE, callsign, lastHour, starttime, kmlfh, AltRE, fly_alt, fly_tilt, fly_heading, fly_range
    match =  dataRE.match(data)

    if not match:
        return False

    latitude = lat(match.group(3))
    longitude = lon(match.group(4))
    hour = int(match.group(2)[:2])
    minute = int(match.group(2)[2:4])
    second = int(match.group(2)[4:])
    
    altitude = int(int(match.group(5))/3.2808)
    if altitude == None or altitude < 100:
        return False

    if altitude > 300:
        altitude = 300
    print 'fly'+ str(match.groups())+"\n"
    print 'Fly: longitude>%s latitude>%s  altitude>%s  \n'%(str(longitude),str(latitude), str(altitude))
    kmlfh.write("<gx:FlyTo>\n")
    kmlfh.write("<gx:duration>2.0</gx:duration>\n")
    kmlfh.write("<gx:flyToMode>smooth</gx:flyToMode>\n")
    kmlfh.write("<LookAt>\n")
    kmlfh.write ('<longitude>%s</longitude> <latitude>%s</latitude>  <altitude>%s</altitude>  <range>%d</range>   <tilt>%d</tilt> <heading>%d</heading> <altitudeMode>relativeToGround</altitudeMode>'%(str(longitude),str(latitude), fly_alt, fly_range, fly_tilt, fly_heading) + "</LookAt> </gx:FlyTo>\n ")
        
    return True

#####
# main
#####
parser = OptionParser(
'''
radioshield2kml.py reads APRS packets captured by a Argent Data System Radio Shield.  The Radio Shield does not write a complete APRS record.  The path data is trunkated.

Assume you are not logging Parachutists jumping from 13,000ft, make sure the set the --min_altitude option to 0.

If you want to tweek this script, you will probalby need to change dataRE and tactical_data().  Possibly the handling of the date stamps if you change from hour minute second.  Maybe processFly as well.

dataRE is the recular expression tht parses the APRS line.
tactical_data() returns the string to be displayed in the bubble when you click on a datapoint.
processFly() is the location to start the point of view.  Depending on your event, altitude or distance will need tweeking.

For the --fly-? options, read this http://code.google.com/apis/kml/documentation/kmlreference.html#lookat
These options feed the altitude, heading, tilt and range elements of the lookat for the FlyTo element setting the initial point of view.

'''
)
parser.add_option('', '--aprs', dest='aprsfi',
                  help='Aprs File to Read', metavar='FILE')
parser.add_option('', '--kml', dest='kmlfi',
                  help='KML File to Write', metavar='FILE', default=None)
parser.add_option('', '--startdate', dest='starttime',
                  help='startdate mmddyyyy', type='string', default =None)
parser.add_option('', '--callsign', dest='callsign',
                  help='Callsign to extract.', type='string')
parser.add_option('', '--min-altitude', dest='min_altitude', default=min_altitude,
                help='The minimum altutude we care about.  Event below this are igored.  Default is %d' % min_altitude,
                 type='int')
parser.add_option('', '--fly-alt', dest='fly_alt', default=fly_alt, type='int',
                help='The Altitude used for the initial position')
parser.add_option('', '--fly-tilt', dest='fly_tilt', default=fly_tilt, type='int',
                help='The Tilt used for the initial position')
parser.add_option('', '--fly-heading', dest='fly_heading', default=fly_heading, type='int',
                help='The heading used for the initial position')
parser.add_option('', '--fly-range', dest='fly_range', default=fly_range, type='int',
                help='The initial distance from the point')

(options, args) = parser.parse_args()


min_altitude = options.min_altitude

# the aprs data file must exist
if options.aprsfi is None:
    print "--aprs required"
    parser.print_help()
    sys.exit(1)
if not os.path.exists(options.aprsfi):
    print options.aprsfi+' does not exist'
    sys.exit(1)

callsign = options.callsign

# the output fule defautls to callsign.kml
# unless a output file name is provided
kmlfi = options.callsign+'.kml'
if options.kmlfi:
    kmlfi = options.kmlfi

#Since we use just time stamp, a date is require to format the date stamp for KML
if not options.starttime:
    print '--startdate is required and the format must be mmddyyyy'
    exit(-1)
starttime= options.starttime

#flyTo parameters
fly_alt = options.fly_alt
fly_tilt = options.fly_tilt
fly_heading = options.fly_heading
fly_range = options.fly_range
# lets process some data

kmlfh = open(kmlfi, 'wb')

#header
kmlfh.write('<?xml version="1.0" encoding="UTF-8"?>'+"\n")
kmlfh.write('<kml xmlns="http://www.opengis.net/kml/2.2"'+"\n")
kmlfh.write('  xmlns:gx="http://www.google.com/kml/ext/2.2">'+"\n")
kmlfh.write('<!--'+"\n")
kmlfh.write('TimeStamp is recommended for Point.'+"\n")
kmlfh.write("Each Point represents a sample from a GPS.\n")
kmlfh.write('-->'+"\n")
kmlfh.write('<Document>'+"\n")
kmlfh.write('<open>1</open>'+"\n")

kmlfh.write('<gx:Tour>'+"\n")
kmlfh.write('<gx:Playlist>'+"\n")

# try to find the FlyTo data point
fi = open(options.aprsfi, 'rb')
while     True:
    line = fi.readline()
    if not line: #end of file
        break
    if line.lower().startswith(options.callsign.lower()):
        if processFly(line):
            break # After the first one we are done.
fi.close()

#kmlfh.write(' </gx:Playlist>')
#kmlfh.write(' </gx:Tour>')
#kmlfh.write('<Folder>'+"\n")

#kmlfh.write('<name>Points with TimeStamps</name>'+"\n")

#style info
kmlfh.write('<Style id="myDefaultStyles">'+"\n")
#icon default
kmlfh.write("\t<IconStyle>\n")
kmlfh.write("\t\t<Icon>\n")
kmlfh.write("\t\t\t<href>%s</href>\n"%parachute_img)
kmlfh.write("\t\t"+'<hotSpot x="70" y="140" xunits="pixels" yunits="pixels"/>'+"\n")
kmlfh.write("\t\t</Icon>\n")
kmlfh.write("\t</IconStyle>\n")

kmlfh.write('<BalloonStyle>'+"\n")
kmlfh.write('<text><![CDATA[')
kmlfh.write('$[description]'+"\n")
kmlfh.write(']]></text>'+"\n")
kmlfh.write('</BalloonStyle>'+"\n")

kmlfh.write('</Style>'+"\n")

# plane icon style
kmlfh.write('<Style id="myPlaneStyles">'+"\n")
kmlfh.write("\t<IconStyle>\n")
kmlfh.write("\t\t<Icon>\n")
kmlfh.write("\t\t\t<href>%s</href>\n"%plane_img)
kmlfh.write("\t\t"+'<hotSpot x="70" y="140" xunits="pixels" yunits="pixels"/>'+"\n")
kmlfh.write("\t\t</Icon>\n")
kmlfh.write("\t</IconStyle>\n")

kmlfh.write('<BalloonStyle>'+"\n")
kmlfh.write('<text><![CDATA[')
kmlfh.write('$[description]'+"\n")
kmlfh.write(']]></text>'+"\n")
kmlfh.write('</BalloonStyle>'+"\n")

kmlfh.write('</Style>'+"\n")

# read the file again to generate all the data points

fi = open(options.aprsfi, 'rb')
while     True:
    line = fi.readline()
    if not line: 
        break
    if line.lower().startswith(options.callsign.lower()):
        processline(line)

#close it out
#kmlfh.write('</Folder>'+"\n")
kmlfh.write(' </gx:Playlist>\n')
kmlfh.write(' </gx:Tour>\n')

kmlfh.write('</Document>'+"\n"+'</kml>')
kmlfh.close()
