// SWR Scanner 1.01 Copyright 2012 by Alan Biocca W6AKB www.akbeng.com w6akb@arrl.net
//
// This program is distributed under the terms of the GNU General Public License

// tested with Arduino software version 0023 on UNO board
// using LCDKeyPadShield v2.0
// with SWR bridge Vr to A1, Vf to A2
// with AmQRP DDS-60 on pins P1-1,2,3 to D2,11,8
//
// rotary encoder knob not yet
// graphical display not yet

#define START 3000000L   // low frequency limit 3 mhz, min 1 mhz
#define END  30000000L   // high frequency limit 30 mhz, max 60 mhz


#include <LCD4Bit_mod.h>           // include LCD library
LCD4Bit_mod lcd = LCD4Bit_mod(2);  // initialize 2 line LCD

void kbd_test()  // Test LCD and Keyboard
{
  int key;
  
  lcd.cursorTo(1,0);
  lcd.printIn("Keyboard Test   ");

  analogReference(DEFAULT);         // keyboard test (requires default analog reference 5V)
  key = analogRead(0);

  lcd.cursorTo(2, 0);               // display value on second line
  
  prInt(key);                       // display adc value
  lcd.printIn("                ");  // clear rest of line
} 


// DDS-60 config, where the I/O is connected
#define DDSCLOCK 11
#define DDSDATA 8
#define DDSLOAD 2

void DDS_init()
{
  pinMode(DDSCLOCK,OUTPUT);  // set pins to output mode
  pinMode(DDSDATA,OUTPUT);
  pinMode(DDSLOAD,OUTPUT);
}

void DDS_pintest()
// output millisecond level signals on DDS pins for scope testing
// unplug DDS for these tests
{
  lcd.cursorTo(1,0);
  lcd.printIn("DDS PinTest     ");
  digitalWrite(DDSDATA,HIGH);
  digitalWrite(DDSCLOCK,HIGH);
  digitalWrite(DDSLOAD,HIGH);
  delay(10);
  digitalWrite(DDSDATA,LOW);   // 10ms on data DDS p3
  delay(10);
  digitalWrite(DDSCLOCK,LOW);  // 20ms on clock DDS p2
  delay(10);
  digitalWrite(DDSLOAD,LOW);   // 30ms on load DDS p1
  delay(10);
}

void DDS_test_alt()  // alternate between two frequencies
{
  lcd.cursorTo(1,0);
  lcd.printIn("DDS 7150/7140   ");
  DDS_freq(7150000);
  delay(1000);
  DDS_freq(7140000);
  delay(500);
}

void DDS_test_onoff()  // turn carrier on and off
{
  lcd.cursorTo(1,0);
  lcd.printIn("DDS 7150 OnOff  ");
  DDS_freq(7150000);
  delay(1000);
  DDS_off();
  delay(500);
}


void DDS_freq(unsigned long freq)  // set DDS freq in hz
{
  unsigned long clock = 180000000L;  // 180 mhz, may need calibration
  int64_t scale = 0x100000000LL;
  unsigned long tune;                // tune word is delta phase per clock
  int ctrl,i;                        // control bits and phase angle
  
  if (freq == 0)
  {
    tune = 0;
    ctrl = 4;  // enable power down
  }
  else
  {
    tune = (freq * scale) / clock;
    ctrl = 1;  // enable clock multiplier
    
    digitalWrite(DDSLOAD, LOW);  // reset load pin
    digitalWrite(DDSCLOCK,LOW);  // data xfer on low to high so start low
    
    for (i = 32; i > 0; --i)  // send 32 bit tune word to DDS shift register
    {
      digitalWrite(DDSDATA,tune&1);  // present the data bit
      digitalWrite(DDSCLOCK,HIGH);   // clock in the data bit
      digitalWrite(DDSCLOCK,LOW);    // reset clock
      tune >>= 1;                    // go to next bit
    }
  }
  for (i = 8; i > 0; --i)  // send control byte to DDS
  {
    digitalWrite(DDSDATA,ctrl&1);
    digitalWrite(DDSCLOCK,HIGH);
    digitalWrite(DDSCLOCK,LOW);
    ctrl >>= 1;      
  }
  // DDS load data into operating register
  digitalWrite(DDSLOAD, HIGH);  // data loads on low to high
  digitalWrite(DDSLOAD, LOW);   // reset load pin
}

void DDS_off()  // shut down DDS
{
  DDS_freq(0L);
}


int vf,vr;  // forward and reflected voltage variables

void swr_test()  // measure Vf Vr and display them and computed SWR
{
  lcd.cursorTo(1,0);
  lcd.printIn(" 7.150 SWR Test ");
  
  int s = swr(7150000);    // set frequency
  lcd.cursorTo(2,0);       // update display
  print_spc();
  prInt(vf);               // display forward voltage ADC counts
  print_spc();
  prInt(vr);               // display reverse voltage ADC counts
  print_spc();
  print_swr(s);            // compute and display swr
  lcd.printIn("      ");
}

void prInt(unsigned int v)
{
  if (v > 9) prInt(v/10);
  lcd.print((v%10)+48);
}

int swr(long f)  // measure SWR, return SWR*10
{
  DDS_freq(f);                           // set the frequency
  delay(1);                              // allow a millisecond for things to settle
  vf = analogRead(2);                    // read analog forward voltage
  vr = analogRead(1);                    // read analog reflected voltage
  if (f == 0 || vr >= vf) return 99;     // limit result to 9.9:1
  int swr = ((vf + vr)*10) / (vf - vr);  // calculate swr times 10
  if (swr > 99) return 99;               // insure value is in range
  return swr;                            // return result
}

long freq=0,low=0,high=0;  // frequency variables


void swr_search()  // find the 3:1 swr region
{
  lcd.cursorTo(1,0);
  lcd.printIn("SWR Search 3:1  ");
  low = high = 0;
  for (freq = START; freq <= END; freq += (freq>>11))  // scan over the frequency range
  {
    int s = swr(freq);  // set frequency and read SWR, return is 10x SWR
  
    if (s < 30)  // if swr below 3:1
    {
      high = freq;
      if (low == 0) low = freq;  // grab first low swr frequency
    }
    else  // swr is above 3:1
    {
      if (high)  // passed through a dip
      {
        lcd.cursorTo(2,0);  // display the region found
        print_mhz(low/1000);
        lcd.printIn("-");
        print_mhz(high/1000);
        lcd.printIn("        ");
        break;
      }
    }
  }
  DDS_off();
  if (high <= low)  // if we didn't find a dip then clear the display line
  {
    lcd.cursorTo(2,0);
    lcd.printIn("                ");
  }
}


void print_bw()  // compute and print bandwidth in khz
{
  int bw = (high-low)/1000;
  if (bw > 999) bw = 999;
  if (bw < 3) bw = 0;
  if (bw < 100) print_spc();
  if (bw < 10) print_spc();
  prInt(bw);
}

void print_spc()
{
  lcd.printIn(" ");
}

void print_mhz(long f)  // print freq using format xx.xxx mhz
{
  int i = f/1000;
  if (i < 10) print_spc();
  prInt((int)(f/1000));
  lcd.printIn(".");
  i = f%1000;
  if (i < 100) lcd.printIn("0");
  if (i < 10) lcd.printIn("0");
  prInt(i);
}

void print_swr(int s)  // print swr using format 9.9:1
{
  prInt(s/10);
  lcd.printIn(".");
  prInt(s%10);
  lcd.printIn(":1");
}

int s;            // swr
int s_best = 99;  // best swr
long f_best;      // freq of best swr

void refine_minswr()  // refine the minimum swr value in this neighborhood
{
  low = high = f_best;   // start the scan at the best spot
  s_best = swr(f_best);  // verify the swr at the current best spot
  
  if (s_best > 18)       // if not a great SWR then look harder
  {
    int i = f_best>>7;   // scan the neighborhood a bit more thoroughly
    for (freq = f_best-i; freq <= (f_best+i); freq += freq>>11)
    {
      s = swr(freq);
      if (s < s_best)
      {
        s_best = s;
        f_best = freq;
      }
    }
  }
  
  while (1)  // first search down by khz steps
  {
    low -= 1000;         // march down
    s = swr(low);        // measure the swr
    if (s > 20) break;   // quit when we hit 2:1 swr
    if (s < s_best)      // if found better
    {
      s_best = s;        // keep it
      f_best = low;
    }
  }
  
  while (1)  // then search up
  {
    high += 1000;
    s = swr(high);
    if (s > 20) break;
    if (s < s_best)
    {
      s_best = s;
      f_best = high;
    }
  }
}

void smart_search()  // scan for first minimum swr point, bottom up
{
  lcd.cursorTo(1,0);
  lcd.printIn("FcMhz BwKhz SWR ");
  
  if (s_best > 35)  // do fast search if we dont know a neighborhood to look in
  {
    for (freq = START; freq <= END; freq += (freq>>7))  // fast search bottom up
    {
      s = swr(freq);
      if (s < s_best)  // if better
      {
        s_best = s;    // remember it
        f_best = freq;
      }
      if (s_best <= 35 && s > 45) break;  // if bad but was better then found first dip then break
    }
  }
  
  if (s_best <= 35) refine_minswr();  // if SWR 3.5:1 or better then search the neighborhood
  
  DDS_off();  // shut DDS off to save power and reduce interference
  lcd.cursorTo(2,0);
  
  if (s_best > 35)  // if SWR was poor blank the display and re-do fast search next time
  {
    lcd.printIn("                ");  // clear lcd line
  }
  else  // swr dip was decent so show it
  {     
      print_mhz(f_best/1000);  // display FcMhz BwKhz SWR
      print_spc();
      print_bw();
      print_spc();
      print_swr(s_best);
  }
}


void setup()  // Arduino user code "sketch" initialization is called once
{
  DDS_init();
  lcd.init();
  lcd.clear();
  lcd.printIn("SWR Scanner 1.00");
  lcd.cursorTo(2,0);
  lcd.printIn("by Alan B  W6AKB");
  delay(3000);
  analogReference(INTERNAL);  // set to low voltage range for better SWR readings
}


void user_program()  // first user program, edit this function
{
  lcd.clear();  // erase the LCD and set the cursor to first line, first position
  lcd.printIn("change this");  // put up to 16 characters of text between the quotes
  lcd.cursorTo(2,0);  // position to second line of LCD, first position
  lcd.printIn("and this"); // put up to 16 characters of text between the quotes
  delay(1000);
}


void loop()  // Arduino user code "sketch" - loop code is called repeatedly after initialization
{
  // uncomment ONE of the following tests at a time (leave the delay below as is)

//  kbd_test();        // test LCD and keyboard, display analog value read from LCD keyboard

//  user_program();    // first user program function, edit the user program function above

//  DDS_pintest();     // send millisecond signals out the DDS pins for scoping, unplug DDS

//  DDS_test_alt();    // alternate between 7150 and 7140 khz carrier

//  DDS_test_onoff();  // on-off key 7150 khz carrier

//  swr_test();        // test swr bridge; read and display Vf Vr SWR at 7150 khz

//  swr_search();      // search for 3:1 swr window

  smart_search();     // search for first swr dip and refine it and measure 2:1 SWR bandwidth

  delay(100);         // 100 millisecond delay between repeats of the above functions
}

// eof de w6akb 73
