// Cascata, An Arduino Waterfall program for ham radio use
// Copyright 2011-2012 Leigh L. Klotz, Jr.
// WA5ZNU May 30, 2011 and May, 30 2012
// Display code based on Mark Sproul LCD library, with my additions
// FFT Code is Roberts/Slaney/Bourras/DEIF FFT library with bug fixes in 8-bit conversion.
// Ideas also from http://blurtime.blogspot.com/2010/11/arduino-realtime-audio-spectrum.html
// License: http://www.opensource.org/licenses/mit-license

#include <Arduino.h>
#include "fix_fft.h"
#include "LCD_driver_progmem.h"
#include "logtable.h"

// Audio Options
#define AOC_THRESHOLD (1)

// Sampling Options
#define SAMPLE_US_8KHZ (125)
#define SAMPLE_US_4KHZ (250)
#define IN_SAMPLES (256)
#define LOG2_IN_SAMPLES (8)
#define OUTBINS (IN_SAMPLES/2)
#define ADC_PIN (1)

// Colors and locations
#define BGCOLOR (BLACK)
#define SPECTRUM_COLOR (GREEN)
#define SPECTRUM_Y_OFFSET (127)

// data is signed going into FFT, signed coming out of FFT, but byte once we compute log power.
char data[IN_SAMPLES];
char im[IN_SAMPLES];

// Attenuator.  ADC samples shifted right by this number, to divide by 2^n.
// Using a value of at least 1 cleans up ADC noise without using CPU sleep.
// Low levels of audio input will require attenuation=0.  Use unsigned
// so we can detect underflow and wrap value.
#define MAX_ATTENUATION (4)
char attenuation = 1;

// Pallette.
byte palette[16];		// could be program memory

// Buttons modes
struct Modes {
  char *name;
  void (*pushed)(byte dir);
};

byte mode = 0;
#define NMODES (3)
struct Modes modes[NMODES] = {
  { "Disp", &handleDisplayButton  },
  { "BW  ", &handleSampleRateButton },
  { "Attn", &handleAttenuatorButton }  
};


int showWaterfall = 1;


int showSpectrum = 1;
byte lastpass[OUTBINS];

int adc_offset = 512;  // assume DC component = 0.  AOC will correct it.
int sample_sum = 0;   // scale/2 to fit in short; maxval=1023, max samples=128, so scale maxval/2.
byte sample_us = SAMPLE_US_8KHZ;

void setup() {
  ioinit();
  LCDInit();
  LCDContrast(44);
  analogReference(INTERNAL); // INTERNAL=1.1v on Arduino UNO

  initPallette();
  clearLastPass(127);
  repaint();
  {
    LCDPutLin("Cascata", 0, 52, SPECTRUM_COLOR, BGCOLOR);
    LCDPutLin("WA5ZNU", 16, 52, SPECTRUM_COLOR, BGCOLOR);
    delay(1000);
    repaint();
  }
}

static inline int calculatePixel(byte i) {
  byte color = ((byte *)data)[i];
  if (color > 15) 
    return 0xf00;  // Full RED
  else
    return palette[color];
}

void initPallette() {
  for (byte i = 0; i < 16; i++) {
    if (i > 9)
      palette[i] = i * 0x110;  // Yellow scale
    else if (i > 5)
      palette[i] = i * 0x010;  // Green Scale
    else
      palette[i] = i * 0x001;  // Blue Scale
  }
}

void loop() {
  static int row = 0;
  static int sample = 0;
  static unsigned long ttt;
  unsigned long next = micros();
  if (next-ttt < sample_us) 
    return;
  ttt = next;
  if (sample < IN_SAMPLES) {
    int val = (analogRead(ADC_PIN) - adc_offset);
    val = val >> attenuation;
    sample_sum += val;
    data[sample] = constrain(val, -128, 127);
    im[sample] = 0;
    sample++;  
  } else {
    // The FFT and display takes way longer than one sample to run, but skipping
    // samples is OK since it's just for display.  If we were doing decoding of data
    // it would be a problem.

    handleButtons();
    {
      // Automatic Offset Correction finds zero level for DC offset of ADC
      // Average of samples should be near zero.
      // 2^7 samples but already scaled by 1/2 so divide sum by 2^(7-1)
      short sample_avg = sample_sum >> (7-1); 
      if (abs(sample_avg) > AOC_THRESHOLD) {
	adc_offset += sample_avg >> 2;  // correct over two rows
      }
      sample_sum = 0;
    }

    sample=0;
    fix_fft(data,im,LOG2_IN_SAMPLES,0);
    // start at bin 1 because bin 0 is DC anyway
    for (byte i=1; i < OUTBINS; i++) {
      // calculate power in bins; convert to dB; 10x instead of 10x to eliminate sqrt.
      // max power value is -128*-128*2=32768 10log10(32768)= 45.15
      // For spectrum was draw the whole thing, since even 45 pixels isn't too much to fit on the screen,
      // but it's slow to draw, so we want to encourage smaller values,
      // so for waterfall we just use the range [0,15] and anything above that is "overrange."
      // we accept up to about 500 on the spectrum.
      ((byte *)data)[i] =db(data[i] * data[i] + im[i] * im[i]);
    }
    if (showSpectrum) {
      for (byte i=1; i<OUTBINS; i++) {
        byte d = ((byte *)data)[i];
        byte y = SPECTRUM_Y_OFFSET-d;
        byte lasty = lastpass[i];
        if (y != lasty) {
          if (y > lasty) {
            LCDSetLine(SPECTRUM_Y_OFFSET,i,lasty,i,BGCOLOR);
          }
          LCDSetLine(SPECTRUM_Y_OFFSET,i,y,i,SPECTRUM_COLOR);
          lastpass[i]=y;
        }
      }
    }
    if (showWaterfall) {
      {
	LCDStartPixelArea(row, 1, row, OUTBINS-1);
	for (byte i = 0; i < OUTBINS; i++) {
	  LCDSetNextPixel(calculatePixel(i));
	}
	LCDEndPixelArea(row, OUTBINS-1);
      }
      row = (row+1) & 0x7F;
      if (row == 0) {
        repaint();
      }
    }
  }
}

void clearLastPass(byte v) {
  if (showSpectrum) {
    for (byte i=0; i<OUTBINS; i++) {
      lastpass[i]=v;
    }
  }
}

// Button 1: Step through modes
// Button 2: +1 on mode.  Wraps around.
// Button 3: -1 on mode.  (Button 3 is not available on some LCD boards.)
void handleButtons() {
  if (!digitalRead(kSwitch1_PIN)) {
    handleModeButton(0);
    modes[mode].pushed(-1);
    while (!digitalRead(kSwitch1_PIN)) delay(20);
  } else if (!digitalRead(kSwitch2_PIN)) {
    handleModeButton(0);
    modes[mode].pushed(1);
    while (!digitalRead(kSwitch2_PIN)) delay(20);
  } 
  else if (!digitalRead(kSwitch3_PIN)) {
    handleModeButton(1);
    while (!digitalRead(kSwitch3_PIN)) delay(20);
  }
}

// Button 1: Both (default), Waterafll only, Spectrum Only
void handleDisplayButton(byte dir) {
  static byte displayFlag = 1|2;
  displayFlag = (displayFlag+1) % 4;
  if (displayFlag == 0) displayFlag = 1;
  showWaterfall = (displayFlag & 1) != 0;
  showSpectrum = (displayFlag & 2) != 0;
  repaint();
}

// 0-4 kHz (default) or 0-2 kHz
void handleSampleRateButton(byte dir) {
  char * msg;
  if (sample_us == SAMPLE_US_4KHZ) {
    msg ="0-4 kHz";
    sample_us = SAMPLE_US_8KHZ;
  } else {
    msg = "0-2 kHz";
    sample_us = SAMPLE_US_4KHZ;
  }
  LCDPutLin(msg, 16, 52, SPECTRUM_COLOR, BGCOLOR);
}

// Attenuator: 0, 1, 2, 3.  Each is a factor of 2.

void handleAttenuatorButton(byte dir) {
  attenuation += dir;
  if (attenuation <= -MAX_ATTENUATION) 
    attenuation = MAX_ATTENUATION-1;
  else if (attenuation >= MAX_ATTENUATION) 
    attenuation = -MAX_ATTENUATION;
  char msg[5]; // length("-255\0") = 5
  itoa(attenuation, msg, 10);
  LCDPutLin(msg, 16, 52, SPECTRUM_COLOR, BGCOLOR);
}

void repaint() {
  LCDClear(BGCOLOR);
}

void handleModeButton(char dir) {
  mode = mode + dir;
  mode = mode % NMODES;
  LCDPutLin(modes[mode].name, 0, 52, SPECTRUM_COLOR, BGCOLOR);
}
