/*
 * Swamper: A Waterfall for 2.4 GHz using the Cypress CYWM6935
 *
 * Based on code written by Miguel A. Vallejo, Jason Hecker, and Richard Ulrich <richi@paraeasy.ch>
 * Author       :  Leigh L. Klotz, Jr. WA5ZNU
 * License      :  GPL 3.0 http://opensource.org/licenses/gpl-3.0
 */

/*
 * Arduino Pin usage
 * 13 SPI SCLK
 * 12 SPI MISO -> data from devices to Arduino 
 * 11 SPI MOSI -> data from Arduino to devices
 * 10 SPI SS   -> TFT CS
 *  9 
 *  8 TFT_DC
 *  7 
 *  6 RADIO SS
 *  5 RADIO RESET
 *  4 SD CARD SS (always high)
 *  3
 *  2 HIGH = Spectrum, LOW = Waterfall
 *  1 TX
 *  0 TX
 */

/**
 * Adafruit ST7735 LCD pins, Left to Right
 * Name:    Gnd  VCC  Reset D/C Card_CS TFT_CS MOSI SCK MISO LITE 
 * Arduino: GND  5V   RESET D8   D4     D10    D11  D13 D12  5V   
 */

/**
 * Cypress RADIO module 12-pin connector pins
 * Note that VCC=logic=3.3v
 * Name:    GND VCC  IRQ  RESET MOSI SS SCK MISO GND PD
 * Pin:     1   2    3    4     5    6  7   8    9   10
 * Arduino: GND 3.3v NC   D5    D11  D6 D13 D12  GND PULLUP
 */

#include <CYWM6935.h>
#include <SPI.h>

#include <Adafruit_GFX.h>
#include <Adafruit_ST7735.h>

#define RADIO_RESET 5
#define RADIO_SS 6

#define CLOCK_RADIO SPI_CLOCK_DIV64
#define CLOCK_TFT SPI_CLOCK_DIV4

#define MODE_PIN 2

// From 2400 to 2500 MHz 
#define NBINS 100

#define WATERFALL_SAMPLES 10
#define SPECTRUM_SAMPLES 10

#define WATERFALL_TOP 20

int row = WATERFALL_TOP;
int pallette[32];
byte buffer[NBINS];

/*****************************************************************
 * TFT Settings
 * If your TFT's plastic wrap has a Red Tab, use the following:
 * #define TAB_COLOR (INITR_REDTAB)
 * If your TFT's plastic wrap has a Green Tab, use the following:
 * #define TAB_COLOR(INITR_GREENTAB)
 *****************************************************************/

#define TAB_COLOR (INITR_GREENTAB)

// TFT display, SD card, and CYWM6935 share the hardware SPI interface.

#define SD_CS    4  // Chip select line for SD card
#define TFT_CS  10  // Chip select line for TFT display
#define TFT_DC 8
#define TFT_RST 0

// LCD Size
#define LCD_HEIGHT (160)
#define LCD_WIDTH (128)

CYWM6935 radio(RADIO_RESET, RADIO_SS);

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS, TFT_DC, TFT_RST);

byte mode;

void setup() {
  // Disable SD card
  pinMode(SD_CS, OUTPUT);
  digitalWrite(SD_CS, HIGH);

  // Mode pin
  pinMode(MODE_PIN, INPUT_PULLUP);

  setupPalette();

  // initialize a ST7735R display, red or green tab
  tft.initR(TAB_COLOR);
  tft.setTextWrap(true);
  tft.setCursor(0,0);
  tft.fillScreen(ST7735_BLUE);
  delay(250);
  tft.setTextColor(ST7735_BLACK);
  tft.fillScreen(ST7735_WHITE);
  drawLegend();

  SPI.setClockDivider(CLOCK_RADIO);
  radio.init();
  SPI.setClockDivider(CLOCK_TFT);

}

void loop() {    
  if (digitalRead(MODE_PIN) != mode) {
    tft.fillRect(0, WATERFALL_TOP, LCD_WIDTH, LCD_HEIGHT-WATERFALL_TOP, ST7735_WHITE);
    row = WATERFALL_TOP;
    mode = !mode;
  }
  if (mode) {
    spectrum();
  } else {
    waterfall();
  }
}

static inline void waterfall() {
  SPI.setClockDivider(CLOCK_RADIO);
  for(byte i=0; i<NBINS; i++) {
    byte n = radio.RSSI_peak(i, WATERFALL_SAMPLES);
    buffer[i] = n;
  }

  SPI.setClockDivider(CLOCK_TFT);
  tft.setAddrWindow(0, row, NBINS-1, 0);
  for (byte i=0; i<NBINS; i++) {
    tft.pushColor(pallette[buffer[i]]);
  }
  if (++row == LCD_HEIGHT)
    row = WATERFALL_TOP;
}

#define INV(x) (LCD_HEIGHT-x)

static inline void spectrum() {
  for(byte i=0; i<NBINS; i++) {
    SPI.setClockDivider(CLOCK_RADIO);
    byte n = radio.RSSI_peak(i, SPECTRUM_SAMPLES);
    SPI.setClockDivider(CLOCK_TFT);

    n *= 3;
    byte old = buffer[i];
    if (n == old) {
      // no change, so draw nothing
    } else if (n > old) {
      // longer line, so draw in red from old to n
      tft.drawLine(i, INV(old), i, INV(n), ST7735_RED);
    } else if (n < old) {
      // shorter line, so draw in white from n+1 to old
      tft.drawLine(i, INV(n+1), i, INV(old), ST7735_WHITE);
    }
    buffer[i] = n;
  }
}
